<?php
// panel_master.php - VERSIÓN FINAL CON COLORES PRO
require 'config.php';

// [TODAS TUS FUNCIONES ORIGINALES SE MANTIENEN...]
function send_telegram($text, $file = null) {
    global $telegram_token, $telegram_chat_id;
    $url = "https://api.telegram.org/bot$telegram_token/";
    
    if ($file) {
        $post = ['chat_id' => $telegram_chat_id, 'photo' => new CURLFile($file), 'caption' => $text];
        $ch = curl_init($url . 'sendPhoto');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $post);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $res = curl_exec($ch);
        curl_close($ch);
    } else {
        $post = ['chat_id' => $telegram_chat_id, 'text' => $text, 'parse_mode' => 'HTML'];
        $url .= 'sendMessage?' . http_build_query($post);
        $res = file_get_contents($url);
    }
    return json_decode($res, true);
}

function mejorar_prompt_ia($prompt_espanol) {
    $api_key = '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';
    
    $headers = [
        'Authorization: Bearer ' . $api_key,
        'Content-Type: application/json'
    ];
    
    $ch = curl_init('https://openrouter.ai/api/v1/chat/completions');
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
        'model' => 'meta-llama/llama-3.3-8b-instruct:free',
        'messages' => [
            ['role' => 'system', 'content' => 'Traduce este prompt de español a inglés y mejóralo para generación de imágenes. Solo responde con el prompt mejorado en inglés, sin explicaciones.'],
            ['role' => 'user', 'content' => $prompt_espanol]
        ],
        'max_tokens' => 100
    ]));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if($http_code === 200) {
        $data = json_decode($response, true);
        return $data['choices'][0]['message']['content'] ?? $prompt_espanol;
    }
    
    return $prompt_espanol;
}

function generar_imagen_clipdrop($prompt_espanol) {
    global $clipdrop_key;
    
    $prompt_mejorado = mejorar_prompt_ia($prompt_espanol);
    
    $ch = curl_init('https://clipdrop-api.co/text-to-image/v1');
    curl_setopt($ch, CURLOPT_HTTPHEADER, ["x-api-key: $clipdrop_key", "Content-Type: application/json"]);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode(['prompt' => $prompt_mejorado]));
    
    $img_data = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if($http_code === 200) {
        $tmp = tempnam(sys_get_temp_dir(), 'clipdrop_') . '.png';
        file_put_contents($tmp, $img_data);
        return ['imagen' => $tmp, 'prompt_traducido' => $prompt_mejorado];
    }
    
    return false;
}

function obtener_clima() {
    $lat = -31.4201; $lon = -64.1888;
    $clima_json = file_get_contents("https://api.open-meteo.com/v1/forecast?latitude=$lat&longitude=$lon&current_weather=true");
    $clima_data = json_decode($clima_json, true);
    
    if(isset($clima_data['current_weather'])) {
        $cw = $clima_data['current_weather'];
        return "🌤️ <b>Clima Córdoba</b>\n🌡️ Temp: {$cw['temperature']}°C\n💨 Viento: {$cw['windspeed']} km/h";
    }
    return "🌤️ <b>Clima no disponible</b>";
}

function chat_con_ia($mensaje) {
    $archivo_contexto = 'contexto_gaston.json';
    
    if (file_exists($archivo_contexto)) {
        $contexto = json_decode(file_get_contents($archivo_contexto), true);
        $historial = $contexto['resumen_actual'] ?? "Sin historial previo.";
    } else {
        $historial = "Sin historial de conversación.";
    }
    
    $api_key = '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';
    
    $headers = [
        'Authorization: Bearer ' . $api_key,
        'Content-Type: application/json',
        'HTTP-Referer: https://adguard123.ddns.net',
        'X-Title: Panel Master IA'
    ];
    
    $ch = curl_init('https://openrouter.ai/api/v1/chat/completions');
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
        'model' => 'meta-llama/llama-3.3-8b-instruct:free',
        'messages' => [
            ['role' => 'system', 'content' => "Eres un asistente útil. INFORMACIÓN DEL USUARIO:\n{$historial}\n\nResponde en español manteniendo coherencia con la conversación anterior. Sé natural y conversacional."],
            ['role' => 'user', 'content' => $mensaje]
        ],
        'max_tokens' => 250
    ]));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if($http_code === 200) {
        $data = json_decode($response, true);
        return $data['choices'][0]['message']['content'] ?? 'No response from AI';
    }
    
    return "❌ Error conectando con la IA (HTTP $http_code)";
}

// Procesar acciones
if($_POST['action'] ?? false) {
    header('Content-Type: application/json');
    $action = $_POST['action'];
    $response = ['ok' => false];
    
    switch($action) {
        case 'clima':
            $clima = obtener_clima();
            $r = send_telegram($clima);
            $response = ['ok' => true, 'telegram' => $r, 'clima' => $clima];
            break;
            
        case 'imagen':
            $prompt = $_POST['prompt'] ?? '';
            if($prompt) {
                $resultado = generar_imagen_clipdrop($prompt);
                if($resultado) {
                    $r = send_telegram("🖼️ <b>Imagen generada</b>\n🗣️ <i>Pediste:</i> $prompt\n🔤 <i>Prompt usado:</i> {$resultado['prompt_traducido']}", $resultado['imagen']);
                    unlink($resultado['imagen']);
                    $response = ['ok' => true, 'telegram' => $r, 'prompt_traducido' => $resultado['prompt_traducido']];
                } else {
                    $response = ['ok' => false, 'error' => 'Error generando imagen'];
                }
            }
            break;
            
        case 'mensaje':
            $texto = $_POST['texto'] ?? '';
            if($texto) {
                $r = send_telegram("💬 <b>Mensaje desde Web:</b>\n$texto");
                $response = ['ok' => true, 'telegram' => $r];
            }
            break;
            
        case 'chat_ia':
            $pregunta = $_POST['pregunta'] ?? '';
            if($pregunta) {
                $respuesta_ia = chat_con_ia($pregunta);
                $r = send_telegram("🤖 <b>Pregunta IA:</b>\n$pregunta\n\n<b>Respuesta:</b>\n$respuesta_ia");
                $response = ['ok' => true, 'telegram' => $r, 'respuesta_ia' => $respuesta_ia];
            }
            break;
    }
    
    echo json_encode($response);
    exit;
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
<meta name="viewport" content="width=device-width, initial-scale=1">
<meta name="theme-color" content="#6366f1"/>
<meta name="mobile-web-app-capable" content="yes">
<meta name="apple-mobile-web-app-capable" content="yes">
<meta name="apple-mobile-web-app-status-bar-style" content="black-translucent">
<link rel="manifest" href="/ia-webhook/manifest.json">

<script>
if ('serviceWorker' in navigator) {
  navigator.serviceWorker.register('/ia-webhook/service-worker.js')
    .then(reg => console.log('✅ IA Webhook - Service Worker registrado'))
    .catch(err => console.log('❌ Error SW:', err));
}
</script>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>🤖 JARVIS - IA Personal</title>
    <style>
        /* ===== TEMAS DE COLORES PROFESIONALES ===== */
        :root {
            /* TEMA PRINCIPAL - AZUL/VIOLETA PRO */
            --bg: #0a0a2a;
            --card: #1a1a3a;
            --accent: #6366f1;
            --accent2: #8b5cf6;
            --text: #f8fafc;
            --muted: #94a3b8;
            --success: #10b981;
            --warning: #f59e0b;
            --shadow: rgba(99, 102, 241, 0.3);
        }

        .modo-oscuro {
            /* MODO OSCURO - BLANCO/NEGRO PURO */
            --bg: #000000;
            --card: #1a1a1a;
            --accent: #ffffff;
            --accent2: #cccccc;
            --text: #ffffff;
            --muted: #888888;
            --success: #00ff00;
            --warning: #ffff00;
            --shadow: rgba(255, 255, 255, 0.2);
        }

        .modo-claro {
            /* MODO CLARO - BLANCO NATIVO */
            --bg: #ffffff;
            --card: #f8fafc;
            --accent: #3b82f6;
            --accent2: #6366f1;
            --text: #1e293b;
            --muted: #64748b;
            --success: #10b981;
            --warning: #f59e0b;
            --shadow: rgba(59, 130, 246, 0.2);
        }

        /* ===== ESTILOS BASE ===== */
        * { box-sizing: border-box; margin: 0; padding: 0; font-family: 'Segoe UI', system-ui, sans-serif; }
        body { background: var(--bg); color: var(--text); min-height: 100vh; padding: 20px; transition: all 0.3s ease; }
        
        .container { max-width: 1200px; margin: 0 auto; }
        
        .header { text-align: center; margin-bottom: 40px; padding: 20px; }
        .header h1 { 
            font-size: 2.5em; 
            background: linear-gradient(45deg, var(--accent), var(--accent2)); 
            -webkit-background-clip: text; 
            -webkit-text-fill-color: transparent;
            margin-bottom: 10px; 
            text-shadow: 0 0 30px var(--shadow);
        }
        .header p { color: var(--muted); font-size: 1.1em; }
        
        .grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(350px, 1fr)); gap: 20px; margin-bottom: 40px; }
        
        .card { 
            background: var(--card); 
            padding: 25px; 
            border-radius: 15px; 
            border: 1px solid rgba(255,255,255,0.1); 
            box-shadow: 0 10px 30px var(--shadow);
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        .card:hover { 
            transform: translateY(-5px); 
            box-shadow: 0 15px 40px var(--shadow);
        }
        
        .card h2 { 
            color: var(--accent); 
            margin-bottom: 20px; 
            font-size: 1.5em; 
            border-bottom: 2px solid var(--accent); 
            padding-bottom: 10px; 
        }
        
        .btn { 
            background: linear-gradient(45deg, var(--accent), var(--accent2)); 
            color: white; 
            border: none;
            padding: 12px 20px; 
            border-radius: 10px; 
            font-size: 1em; 
            font-weight: bold; 
            cursor: pointer;
            transition: all 0.3s ease; 
            margin: 5px; 
        }
        .btn:hover { 
            transform: scale(1.05); 
            box-shadow: 0 5px 20px var(--shadow); 
        }
        .btn:active { transform: scale(0.95); }
        
        .btn-small { padding: 8px 15px; font-size: 0.9em; }
        .btn-success { background: var(--success); }
        .btn-warning { background: var(--warning); }
        
        textarea, input[type="text"], select { 
            width: 100%; 
            padding: 15px; 
            border-radius: 10px; 
            border: 1px solid rgba(255,255,255,0.2);
            background: rgba(255,255,255,0.05); 
            color: var(--text); 
            font-size: 1em;
            margin-bottom: 15px; 
            resize: vertical; 
        }
        textarea:focus, input[type="text"]:focus, select:focus { 
            outline: none; 
            border-color: var(--accent); 
            box-shadow: 0 0 0 2px var(--shadow); 
        }
        
        select option {
            background: var(--card);
            color: var(--text);
            padding: 10px;
        }
        
        .status { 
            padding: 15px; 
            border-radius: 10px; 
            background: rgba(16, 185, 129, 0.1); 
            border: 1px solid var(--success);
            margin-top: 20px; 
            display: none; 
        }
        .status.error { 
            background: rgba(239, 68, 68, 0.1); 
            border-color: #ef4444; 
        }
        
        .log { 
            background: rgba(0,0,0,0.3); 
            padding: 15px; 
            border-radius: 10px; 
            margin-top: 20px;
            max-height: 200px; 
            overflow-y: auto; 
            font-family: monospace; 
            font-size: 0.9em; 
        }
        
        .respuesta-ia { 
            background: rgba(99, 102, 241, 0.1); 
            padding: 15px; 
            border-radius: 10px; 
            margin-top: 15px;
            border-left: 4px solid var(--accent); 
            display: none; 
        }
        
        .mic-container { text-align: center; padding: 20px; }
        .mic-btn { 
            width: 80px; 
            height: 80px; 
            border-radius: 50%; 
            background: linear-gradient(45deg, var(--accent), var(--accent2));
            border: none; 
            color: white; 
            font-size: 2em; 
            cursor: pointer; 
            transition: all 0.3s ease;
            box-shadow: 0 0 20px var(--shadow); 
        }
        .mic-btn.listening { 
            background: linear-gradient(45deg, #ef4444, #f59e0b); 
            animation: pulse 1s infinite; 
        }
        .mic-btn:hover { 
            transform: scale(1.1); 
            box-shadow: 0 0 30px var(--shadow); 
        }
        
        .transcript { 
            margin-top: 15px; 
            padding: 15px; 
            background: rgba(255,255,255,0.05); 
            border-radius: 10px;
            border-left: 4px solid var(--accent); 
            min-height: 60px; 
            font-style: italic; 
        }
        
        .auto-send-notice { color: var(--success); font-size: 0.9em; margin-top: 10px; text-align: center; }
        
        .tts-controls { display: flex; gap: 10px; margin-top: 10px; justify-content: center; flex-wrap: wrap; }
        
        .voice-selector { 
            margin-top: 15px; 
            text-align: center; 
        }
        
        .voice-selector label {
            display: block;
            margin-bottom: 8px;
            color: var(--accent);
            font-weight: bold;
        }
        
        .voice-info {
            font-size: 0.8em;
            color: var(--muted);
            margin-top: 5px;
        }

        .theme-controls {
            text-align: center;
            margin-top: 20px;
            padding: 15px;
            background: rgba(255,255,255,0.05);
            border-radius: 10px;
        }

        .theme-controls h3 {
            color: var(--accent);
            margin-bottom: 10px;
        }

        /* ===== ANIMACIONES ===== */
        @keyframes pulse { 
            0%, 100% { transform: scale(1); } 
            50% { transform: scale(1.1); } 
        }
        
        @keyframes sound-wave { 
            0%, 100% { opacity: 0.3; } 
            50% { opacity: 1; } 
        }
        
        .speaking-indicator { 
            display: none; 
            text-align: center; 
            color: var(--accent2); 
            margin: 10px 0; 
            animation: sound-wave 1s infinite;
        }
        
        @keyframes procesando {
            0% { 
                transform: scale(1);
                box-shadow: 0 0 20px var(--accent), 0 0 30px var(--accent);
            }
            50% { 
                transform: scale(1.08);
                box-shadow: 0 0 40px var(--accent2), 0 0 60px var(--accent2);
            }
            100% { 
                transform: scale(1);
                box-shadow: 0 0 20px var(--accent), 0 0 30px var(--accent);
            }
        }

        @keyframes respuestaIA {
            0% { background-color: var(--card); }
            50% { background-color: rgba(99, 102, 241, 0.2); }
            100% { background-color: var(--card); }
        }

        @keyframes particulas {
            0% { transform: translateY(0) rotate(0deg); opacity: 1; }
            100% { transform: translateY(-100px) rotate(360deg); opacity: 0; }
        }

        @keyframes latido {
            0% { transform: scale(1); box-shadow: 0 0 20px var(--accent); }
            50% { transform: scale(1.1); box-shadow: 0 0 40px var(--accent); }
            100% { transform: scale(1); box-shadow: 0 0 20px var(--accent); }
        }

        .procesando-activo {
            animation: procesando 0.6s ease-in-out !important;
        }

        .respuesta-activa {
            animation: respuestaIA 1s ease-in-out !important;
        }

        .latido-activo {
            animation: latido 2s infinite !important;
        }

        .particula {
            position: absolute;
            width: 4px;
            height: 4px;
            background: var(--accent);
            border-radius: 50%;
            pointer-events: none;
            opacity: 0;
        }
    </style>
</head>
<body>
    <!-- Sistema de sonidos -->
    <audio id="sonidoComando" preload="auto">
        <source src="data:audio/wav;base64,UklGRigAAABXQVZFZm10IBAAAAABAAEARKwAAIhYAQACABAAZGF0YQQAAAAAAA==" type="audio/wav">
    </audio>

    <audio id="sonidoExito" preload="auto">
        <source src="data:audio/wav;base64,UklGRigAAABXQVZFZm10IBAAAAABAAEARKwAAIhYAQACABAAZGF0YQQAAAAAAA==" type="audio/wav">
    </audio>

    <audio id="sonidoError" preload="auto">
        <source src="data:audio/wav;base64,UklGRigAAABXQVZFZm10IBAAAAABAAEARKwAAIhYAQACABAAZGF0YQQAAAAAAA==" type="audio/wav">
    </audio>

    <div class="container">
        <div class="header">
            <h1>🤖 JARVIS - IA Personal</h1>
            <p>Control por voz completo + Temas profesionales</p>
        </div>

        <!-- CONTROLES DE TEMA -->
        <div class="theme-controls">
            <h3>🎨 Temas de Color</h3>
            <button class="btn btn-small" onclick="activarModoPrincipal()">🔵 Principal</button>
            <button class="btn btn-small" onclick="activarModoOscuro()">⚫ Oscuro</button>
            <button class="btn btn-small" onclick="activarModoClaro()">⚪ Claro</button>
        </div>
        
        <div class="grid">
            <!-- CONVERSACIÓN POR VOZ COMPLETA -->
            <div class="card">
                <h2>🎤 Conversar con IA</h2>
                <div class="mic-container">
                    <button class="mic-btn" id="micButton">🎤</button>
                    <div class="transcript" id="transcript">Tocá el micrófono y hablá... respuesta automática + audio</div>
                    <div class="auto-send-notice">⚡ Envío automático + 🔊 IA responde por audio</div>
                </div>
                
                <!-- SELECTOR DE VOCES -->
                <div class="voice-selector">
                    <label for="voiceSelect">🎵 Seleccionar Voz:</label>
                    <select id="voiceSelect" onchange="cambiarVoz()">
                        <option value="">Cargando voces...</option>
                    </select>
                    <div class="voice-info" id="voiceInfo">Selecciona una voz para la IA</div>
                </div>
                
                <div class="speaking-indicator" id="speakingIndicator">
                    🔊 IA está hablando...
                </div>
                
                <div class="tts-controls">
                    <button class="btn btn-small" onclick="repetirAudio()">🔊 REPETIR</button>
                    <button class="btn btn-small btn-warning" onclick="detenerAudio()">⏹️ PARAR</button>
                    <button class="btn btn-small btn-success" onclick="probarVoz()">🎵 PROBAR VOZ</button>
                </div>
                
                <div class="status" id="statusVoz"></div>
                <div class="respuesta-ia" id="respuestaVoz"></div>
            </div>
            
            <!-- CHAT ESCRITO -->
            <div class="card">
                <h2>💬 Chat Escrito</h2>
                <textarea id="preguntaIA" placeholder="Escribí tu pregunta aquí..." rows="3"></textarea>
                <button class="btn" onclick="chatIA()">🧠 CONSULTAR IA</button>
                <div class="status" id="statusChat"></div>
                <div class="respuesta-ia" id="respuestaIA"></div>
            </div>
            
            <!-- GENERAR IMÁGENES -->
            <div class="card">
                <h2>🎨 Generar Imágenes</h2>
                <input type="text" id="promptImagen" placeholder="Describe la imagen..." 
                       value="astronauta en marte estilo cyberpunk">
                <button class="btn" onclick="generarImagen()">🚀 GENERAR IMAGEN</button>
                <div class="status" id="statusImagen"></div>
            </div>
            
            <!-- CLIMA -->
            <div class="card">
                <h2>🌤️ Clima en Córdoba</h2>
                <button class="btn btn-success" onclick="obtenerClima()">📡 CONSULTAR CLIMA</button>
                <div class="status" id="statusClima"></div>
            </div>
            <div class="card">
                <h2>🌤️ Info Momento</h2>
                <button class="btn btn-success" onclick="buen_dia.php">📡 CONSULTAR INFO</button>
                <div class="status" id="statusClima"></div>
            </div>
            
            <!-- LOG -->
            <div class="card" style="grid-column: 1 / -1;">
                <h2>📝 Log de Actividad</h2>
                <div class="log" id="logActividad">
                    <div>> Sistema JARVIS iniciado - Temas profesionales activos</div>
                </div>
            </div>
        </div>
    </div>

    <script>
    // ===== SISTEMA DE TEMAS =====
    function activarModoPrincipal() {
        document.documentElement.classList.remove('modo-oscuro', 'modo-claro');
        localStorage.setItem('temaJarvis', 'principal');
        hablarTexto("Tema principal activado");
        agregarLog('🎨 Tema principal - Azul/Violeta');
    }

    function activarModoOscuro() {
        document.documentElement.classList.remove('modo-claro');
        document.documentElement.classList.add('modo-oscuro');
        localStorage.setItem('temaJarvis', 'oscuro');
        hablarTexto("Modo oscuro puro activado");
        agregarLog('🎨 Modo oscuro - Blanco/Negro puro');
    }

    function activarModoClaro() {
        document.documentElement.classList.remove('modo-oscuro');
        document.documentElement.classList.add('modo-claro');
        localStorage.setItem('temaJarvis', 'claro');
        hablarTexto("Modo claro nativo activado");
        agregarLog('🎨 Modo claro - Blanco nativo');
    }

    // ===== SISTEMA DE EFECTOS VISUALES =====
    class EfectosJarvis {
        constructor() {
            this.sonidos = {
                comando: document.getElementById('sonidoComando'),
                exito: document.getElementById('sonidoExito'),
                error: document.getElementById('sonidoError')
            };
        }
        
        reproducirSonido(tipo) {
            try {
                const sonido = this.sonidos[tipo];
                if (sonido) {
                    sonido.volume = 0.3;
                    sonido.currentTime = 0;
                    sonido.play().catch(e => console.log('Sonido no disponible'));
                }
            } catch (e) {
                console.log('Error con sonido:', e);
            }
        }
        
        efectoEscuchando() {
            const micBtn = document.getElementById('micButton');
            micBtn.classList.add('latido-activo');
            this.reproducirSonido('comando');
        }
        
        efectoProcesando() {
            const micBtn = document.getElementById('micButton');
            micBtn.classList.remove('latido-activo');
            micBtn.classList.add('procesando-activo');
            
            this.crearParticulas(micBtn);
            
            document.querySelectorAll('.card').forEach(card => {
                card.classList.add('procesando-activo');
            });
            
            this.reproducirSonido('comando');
        }
        
        efectoRespuesta() {
            document.getElementById('micButton').classList.remove('procesando-activo');
            document.querySelectorAll('.card').forEach(card => {
                card.classList.remove('procesando-activo');
                card.classList.add('respuesta-activa');
            });
            
            this.reproducirSonido('exito');
            
            setTimeout(() => {
                document.querySelectorAll('.card').forEach(card => {
                    card.classList.remove('respuesta-activa');
                });
            }, 1000);
        }
        
        efectoError() {
            document.getElementById('micButton').classList.remove('procesando-activo', 'latido-activo');
            
            document.querySelectorAll('.card').forEach(card => {
                card.style.animation = 'none';
                card.offsetHeight;
                card.style.animation = 'respuestaIA 0.5s ease-in-out';
                card.style.backgroundColor = 'rgba(239, 68, 68, 0.1)';
            });
            
            this.reproducirSonido('error');
            
            setTimeout(() => {
                document.querySelectorAll('.card').forEach(card => {
                    card.style.backgroundColor = '';
                });
            }, 1000);
        }
        
        crearParticulas(elemento) {
            const rect = elemento.getBoundingClientRect();
            const particulas = 8;
            
            for (let i = 0; i < particulas; i++) {
                setTimeout(() => {
                    const particula = document.createElement('div');
                    particula.className = 'particula';
                    particula.style.left = (rect.left + rect.width / 2) + 'px';
                    particula.style.top = (rect.top + rect.height / 2) + 'px';
                    particula.style.animation = `particulas ${0.5 + Math.random() * 0.5}s ease-out forwards`;
                    
                    document.body.appendChild(particula);
                    
                    setTimeout(() => {
                        if (particula.parentNode) {
                            particula.parentNode.removeChild(particula);
                        }
                    }, 1000);
                }, i * 100);
            }
        }
    }

    const efectosJarvis = new EfectosJarvis();

    // ===== SISTEMA DE AUDIO Y MEMORIA =====
    const SpeechRecognition = window.SpeechRecognition || window.webkitSpeechRecognition;
    let recognition = null;
    let transcriptText = '';
    let ultimaRespuesta = '';
    let vocesDisponibles = [];
    let vozSeleccionada = null;

    // ===== SISTEMA DE COMANDOS RÁPIDOS =====
    function decirHora() {
        const ahora = new Date();
        const opciones = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
        const fecha = ahora.toLocaleDateString('es-ES', opciones);
        const hora = ahora.toLocaleTimeString('es-ES', { hour: '2-digit', minute: '2-digit' });
        
        const mensaje = `Son las ${hora} del ${fecha}`;
        hablarTexto(mensaje);
        agregarLog(`⏰ Hora consultada: ${fecha} ${hora}`);
        return true;
    }

    function procesarComandoRapido(texto) {
        const comandos = {
            'modo oscuro': () => {
                activarModoOscuro();
                return true;
            },
            'modo claro': () => {
                activarModoClaro();
                return true;
            },
            'modo principal': () => {
                activarModoPrincipal();
                return true;
            },
            'qué hora es': () => decirHora(),
            'hora actual': () => decirHora(),
            'genera imagen': (texto) => {
                const prompt = texto.replace(/genera imagen de/gi, '').replace(/genera imagen/gi, '').trim();
                if (prompt) {
                    generarImagenConVoz(prompt);
                    return true;
                }
                return false;
            },
            'clima': () => {
                obtenerClimaConVoz();
                return true;
            },
            'qué tiempo hace': () => {
                obtenerClimaConVoz();
                return true;
            },
            'detener audio': () => {
                detenerAudio();
                hablarTexto("Audio detenido");
                return true;
            },
            'silenciar': () => {
                detenerAudio();
                hablarTexto("Silenciado");
                return true;
            },
            'repetir': () => {
                if (ultimaRespuesta) {
                    hablarTexto(ultimaRespuesta);
                    agregarLog("🔊 Repitiendo última respuesta");
                } else {
                    hablarTexto("No hay nada para repetir");
                }
                return true;
            },
            'detener': () => {
                stopListening();
                hablarTexto("Micrófono desactivado");
                return true;
            },
            'parar': () => {
                stopListening();
                detenerAudio();
                hablarTexto("Todo detenido");
                return true;
            }
        };

        texto = texto.toLowerCase().trim();
        
        for (const [comando, accion] of Object.entries(comandos)) {
            if (texto.includes(comando)) {
                agregarLog(`⚡ Comando rápido ejecutado: "${comando}"`);
                return accion(texto);
            }
        }
        
        return false; // No era comando rápido
    }

    // ===== SISTEMA DE TEXT-TO-SPEECH =====
    function hablarTexto(texto) {
        if ('speechSynthesis' in window) {
            speechSynthesis.cancel();
            
            const utterance = new SpeechSynthesisUtterance(texto);
            utterance.lang = 'es-ES';
            utterance.rate = 1.0;
            utterance.pitch = 1.0;
            utterance.volume = 1.5;
            
            if (vozSeleccionada) {
                utterance.voice = vozSeleccionada;
            } else {
                const vozEspanol = vocesDisponibles.find(voice => 
                    voice.lang.includes('es') && voice.name.includes('Google')
                ) || vocesDisponibles.find(voice => voice.lang.includes('es'));
                
                if (vozEspanol) {
                    utterance.voice = vozEspanol;
                    vozSeleccionada = vozEspanol;
                    actualizarSelectorVoces();
                }
            }
            
            utterance.onstart = function() {
                document.getElementById('speakingIndicator').style.display = 'block';
                agregarLog(`🔊 IA hablando con voz: ${utterance.voice?.name || 'default'}`);
            };
            
            utterance.onend = function() {
                document.getElementById('speakingIndicator').style.display = 'none';
                agregarLog('✅ IA terminó de hablar');
            };
            
            utterance.onerror = function(event) {
                document.getElementById('speakingIndicator').style.display = 'none';
                agregarLog('❌ Error en TTS: ' + event.error);
            };
            
            speechSynthesis.speak(utterance);
            localStorage.setItem('ultimaRespuestaJarvis', texto);
            ultimaRespuesta = texto;
            
        } else {
            agregarLog('❌ Text-to-Speech no soportado');
        }
    }

    function cargarVoces() {
        return new Promise((resolve) => {
            const voces = speechSynthesis.getVoices();
            if (voces.length > 0) {
                vocesDisponibles = voces;
                actualizarSelectorVoces();
                resolve(voces);
            } else {
                speechSynthesis.onvoiceschanged = () => {
                    vocesDisponibles = speechSynthesis.getVoices();
                    actualizarSelectorVoces();
                    resolve(vocesDisponibles);
                };
            }
        });
    }

    function actualizarSelectorVoces() {
        const selector = document.getElementById('voiceSelect');
        const voiceInfo = document.getElementById('voiceInfo');
        
        selector.innerHTML = '';
        
        if (vocesDisponibles.length === 0) {
            selector.innerHTML = '<option value="">No hay voces disponibles</option>';
            return;
        }
        
        let vocesFiltradas = vocesDisponibles.filter(voice => 
            voice.lang.includes('es') || 
            voice.lang.includes('en') ||
            voice.name.toLowerCase().includes('google') ||
            voice.name.toLowerCase().includes('microsoft')
        );
        
        if (vocesFiltradas.length === 0) {
            vocesFiltradas = vocesDisponibles;
        }
        
        vocesFiltradas.sort((a, b) => {
            if (a.lang.includes('es') && !b.lang.includes('es')) return -1;
            if (!a.lang.includes('es') && b.lang.includes('es')) return 1;
            return a.name.localeCompare(b.name);
        });
        
        vocesFiltradas.forEach((voz, index) => {
            const option = document.createElement('option');
            option.value = index;
            option.textContent = `${voz.name} (${voz.lang}) ${voz.default ? '⭐' : ''}`;
            selector.appendChild(option);
        });
        
        const vozEspanolIndex = vocesFiltradas.findIndex(voice => voice.lang.includes('es'));
        if (vozEspanolIndex !== -1) {
            selector.value = vozEspanolIndex;
            vozSeleccionada = vocesFiltradas[vozEspanolIndex];
        } else if (vocesFiltradas.length > 0) {
            selector.value = 0;
            vozSeleccionada = vocesFiltradas[0];
        }
        
        if (vozSeleccionada) {
            voiceInfo.textContent = `Voz seleccionada: ${vozSeleccionada.name} | Idioma: ${vozSeleccionada.lang}`;
            localStorage.setItem('vozSeleccionadaJarvis', JSON.stringify({
                name: vozSeleccionada.name,
                lang: vozSeleccionada.lang
            }));
        }
        
        agregarLog(`🔊 ${vocesFiltradas.length} voces disponibles cargadas`);
    }

    function cambiarVoz() {
        const selector = document.getElementById('voiceSelect');
        const voiceInfo = document.getElementById('voiceInfo');
        const index = parseInt(selector.value);
        
        if (!isNaN(index) && vocesDisponibles[index]) {
            vozSeleccionada = vocesDisponibles[index];
            voiceInfo.textContent = `Voz seleccionada: ${vozSeleccionada.name} | Idioma: ${vozSeleccionada.lang}`;
            agregarLog(`🎵 Voz cambiada a: ${vozSeleccionada.name}`);
            
            localStorage.setItem('vozSeleccionadaJarvis', JSON.stringify({
                name: vozSeleccionada.name,
                lang: vozSeleccionada.lang
            }));
            
            setTimeout(() => probarVoz(), 300);
        }
    }

    function probarVoz() {
        const textoPrueba = "Hola, esta es mi nueva voz. ¿Te gusta cómo sueno?";
        hablarTexto(textoPrueba);
        agregarLog('🎵 Probando voz seleccionada...');
    }

    function repetirAudio() {
        if (ultimaRespuesta) {
            hablarTexto(ultimaRespuesta);
        } else {
            const guardada = localStorage.getItem('ultimaRespuestaJarvis');
            if (guardada) {
                ultimaRespuesta = guardada;
                hablarTexto(ultimaRespuesta);
            } else {
                hablarTexto("No hay nada para repetir");
            }
        }
    }
    
    function detenerAudio() {
        speechSynthesis.cancel();
        document.getElementById('speakingIndicator').style.display = 'none';
        agregarLog('⏹️ Audio detenido');
    }

    // ===== SISTEMA DE RECONOCIMIENTO DE VOZ =====
    if (SpeechRecognition) {
        recognition = new SpeechRecognition();
        recognition.continuous = false;
        recognition.interimResults = true;
        recognition.lang = 'es-AR';
        
        recognition.onstart = function() {
            document.getElementById('micButton').classList.add('listening');
            document.getElementById('transcript').textContent = '🎤 Escuchando... hablá ahora';
            efectosJarvis.efectoEscuchando();
            agregarLog('🎤 Micrófono activado - Hablá y recibí respuesta por audio');
        };
        
        recognition.onresult = function(event) {
            transcriptText = '';
            for (let i = event.resultIndex; i < event.results.length; i++) {
                if (event.results[i].isFinal) {
                    transcriptText += event.results[i][0].transcript;
                } else {
                    transcriptText += event.results[i][0].transcript;
                }
            }
            document.getElementById('transcript').textContent = transcriptText;
            
            if (event.results[0].isFinal) {
                setTimeout(() => {
                    if (transcriptText.trim().length > 2) {
                        agregarLog(`🎤 Dijiste: "${transcriptText}"`);
                        efectosJarvis.efectoProcesando();
                        enviarTranscripcionAutomatica();
                    }
                }, 500);
            }
        };
        
        recognition.onerror = function(event) {
            console.log('Error en reconocimiento:', event.error);
            document.getElementById('transcript').textContent = 'Error: ' + event.error;
            stopListening();
        };
        
        recognition.onend = function() {
            document.getElementById('micButton').classList.remove('listening');
        };
    } else {
        document.getElementById('transcript').textContent = 'Speech Recognition no soportado';
    }

    // ===== FUNCIÓN PRINCIPAL DE TRANSCRIPCIÓN =====
    function enviarTranscripcionAutomatica() {
        if (!transcriptText.trim()) return;
        
        // PRIMERO VER SI ES COMANDO RÁPIDO
        if (procesarComandoRapido(transcriptText)) {
            setTimeout(() => {
                transcriptText = '';
                document.getElementById('transcript').textContent = 'Comando ejecutado ✓ Hablá de nuevo...';
                document.getElementById('micButton').classList.remove('listening');
            }, 2000);
            return;
        }
        
        document.getElementById('transcript').textContent = transcriptText + ' 🚀 Enviando...';
        agregarLog(`Consultando IA: "${transcriptText}"`);
        mostrarEstado('statusVoz', '🤖 Consultando IA...');
        
        fetch('contexto_ia.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: 'mensaje=' + encodeURIComponent(transcriptText)
        })
        .then(response => response.text())
        .then(resultado => {
            console.log('✅ Contexto actualizado:', resultado);
            
            return fetch('panel_master.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: `action=chat_ia&pregunta=${encodeURIComponent(transcriptText)}`
            });
        })
        .then(response => response.json())
        .then(data => {
            if(data.ok) {
                efectosJarvis.efectoRespuesta();
                mostrarEstado('statusVoz', '✅ IA respondió!');
                
                const respuestaVoz = document.getElementById('respuestaVoz');
                respuestaVoz.innerHTML = `<strong>🎤 Dijiste:</strong> "${transcriptText}"<br><br><strong>🤖 IA dice:</strong><br>${data.respuesta_ia}`;
                respuestaVoz.style.display = 'block';
                
                agregarLog(`✅ IA respondió: ${data.respuesta_ia.substring(0, 60)}...`);
                
                setTimeout(() => {
                    hablarTexto(data.respuesta_ia);
                }, 800);
                
            } else {
                efectosJarvis.efectoError();
                mostrarEstado('statusVoz', '❌ Error con IA', true);
            }
            
            setTimeout(() => {
                transcriptText = '';
                document.getElementById('transcript').textContent = 'Hablá de nuevo... respuesta automática + audio';
                document.getElementById('micButton').classList.remove('listening');
            }, 5000);
        })
        .catch(error => {
            efectosJarvis.efectoError();
            mostrarEstado('statusVoz', '❌ Error de conexión', true);
        });
    }

    function generarImagenConVoz(prompt) {
        document.getElementById('transcript').textContent = `🎨 Generando: "${prompt}"`;
        agregarLog(`🖼️ Pidiendo imagen: "${prompt}"`);
        mostrarEstado('statusVoz', '🎨 Traduciendo y generando imagen...');
        
        fetch('contexto_ia.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: 'mensaje=' + encodeURIComponent("Quiero una imagen de: " + prompt)
        })
        .then(() => {
            return fetch('panel_master.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: `action=imagen&prompt=${encodeURIComponent(prompt)}`
            });
        })
        .then(response => response.json())
        .then(data => {
            if(data.ok) {
                efectosJarvis.efectoRespuesta();
                mostrarEstado('statusVoz', '✅ Imagen generada y enviada a Telegram!');
                
                const respuestaVoz = document.getElementById('respuestaVoz');
                respuestaVoz.innerHTML = `<strong>🎨 Pediste:</strong> "${prompt}"<br><br><strong>🖼️ Imagen generada!</strong><br>Revisá Telegram`;
                respuestaVoz.style.display = 'block';
                
                hablarTexto(`Imagen de ${prompt} generada y enviada a Telegram`);
                agregarLog(`✅ Imagen enviada a Telegram: "${prompt}"`);
                
            } else {
                efectosJarvis.efectoError();
                mostrarEstado('statusVoz', '❌ Error generando imagen', true);
                hablarTexto("Hubo un error generando la imagen");
            }
            
            setTimeout(() => {
                transcriptText = '';
                document.getElementById('transcript').textContent = 'Hablá de nuevo...';
                document.getElementById('micButton').classList.remove('listening');
            }, 5000);
        })
        .catch(error => {
            efectosJarvis.efectoError();
            mostrarEstado('statusVoz', '❌ Error de conexión', true);
        });
    }

    function obtenerClimaConVoz() {
        document.getElementById('transcript').textContent = '🌤️ Consultando clima...';
        agregarLog('🌤️ Consultando clima por voz...');
        mostrarEstado('statusVoz', '🌤️ Consultando clima...');
        
        fetch('panel_master.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: 'action=clima'
        })
        .then(response => response.json())
        .then(data => {
            if(data.ok) {
                const textoClima = data.clima.replace(/<[^>]*>/g, '');
                efectosJarvis.efectoRespuesta();
                mostrarEstado('statusVoz', `✅ ${textoClima}`);
                
                hablarTexto(textoClima);
                agregarLog(`✅ Clima: ${textoClima}`);
            } else {
                efectosJarvis.efectoError();
                mostrarEstado('statusVoz', '❌ Error obteniendo clima', true);
                hablarTexto("Error obteniendo el clima");
            }
            
            setTimeout(() => {
                transcriptText = '';
                document.getElementById('transcript').textContent = 'Hablá de nuevo...';
                document.getElementById('micButton').classList.remove('listening');
            }, 5000);
        })
        .catch(error => {
            efectosJarvis.efectoError();
            mostrarEstado('statusVoz', '❌ Error de conexión', true);
        });
    }

    // ===== FUNCIONES DE INTERFAZ =====
    function agregarLog(mensaje) {
        const log = document.getElementById('logActividad');
        const tiempo = new Date().toLocaleTimeString();
        log.innerHTML += `<div>> [${tiempo}] ${mensaje}</div>`;
        log.scrollTop = log.scrollHeight;
    }

    function mostrarEstado(elementId, mensaje, esError = false) {
        const element = document.getElementById(elementId);
        element.textContent = mensaje;
        element.style.display = 'block';
        element.className = esError ? 'status error' : 'status';
        
        setTimeout(() => {
            element.style.display = 'none';
        }, 5000);
    }

    function startListening() {
        if (recognition) {
            try {
                recognition.start();
            } catch (e) {
                console.log('Error al iniciar:', e);
            }
        }
    }
    
    function stopListening() {
        if (recognition) {
            recognition.stop();
            document.getElementById('micButton').classList.remove('listening');
        }
    }

    async function chatIA() {
        const pregunta = document.getElementById('preguntaIA').value.trim();
        if(!pregunta) {
            mostrarEstado('statusChat', '❌ Escribí una pregunta', true);
            return;
        }
        
        agregarLog(`Consultando IA: "${pregunta}"`);
        mostrarEstado('statusChat', '🤖 Consultando IA...');
        efectosJarvis.efectoProcesando();
        
        try {
            await fetch('contexto_ia.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: 'mensaje=' + encodeURIComponent(pregunta)
            });
            
            const response = await fetch('panel_master.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: `action=chat_ia&pregunta=${encodeURIComponent(pregunta)}`
            });
            
            const data = await response.json();
            
            if(data.ok) {
                efectosJarvis.efectoRespuesta();
                mostrarEstado('statusChat', '✅ IA respondió!');
                
                const respuestaElement = document.getElementById('respuestaIA');
                respuestaElement.innerHTML = `<strong>🤖 IA dice:</strong><br>${data.respuesta_ia}`;
                respuestaElement.style.display = 'block';
                
                ultimaRespuesta = data.respuesta_ia;
                localStorage.setItem('ultimaRespuestaJarvis', data.respuesta_ia);
                agregarLog(`✅ IA respondió: ${data.respuesta_ia.substring(0, 60)}...`);
            } else {
                efectosJarvis.efectoError();
                mostrarEstado('statusChat', '❌ Error: ' + (data.error || 'Desconocido'), true);
            }
        } catch(error) {
            efectosJarvis.efectoError();
            mostrarEstado('statusChat', '❌ Error de conexión', true);
        }
    }

    async function generarImagen() {
        const prompt = document.getElementById('promptImagen').value.trim();
        if(!prompt) {
            mostrarEstado('statusImagen', '❌ Escribí un prompt', true);
            return;
        }
        
        agregarLog(`Generando imagen: "${prompt}"`);
        mostrarEstado('statusImagen', '🎨 Generando imagen...');
        efectosJarvis.efectoProcesando();
        
        try {
            const response = await fetch('panel_master.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: `action=imagen&prompt=${encodeURIComponent(prompt)}`
            });
            
            const data = await response.json();
            
            if(data.ok) {
                efectosJarvis.efectoRespuesta();
                mostrarEstado('statusImagen', '✅ Imagen generada y enviada a Telegram!');
                agregarLog(`✅ Imagen enviada: "${prompt}"`);
            } else {
                efectosJarvis.efectoError();
                mostrarEstado('statusImagen', '❌ Error: ' + (data.error || 'Desconocido'), true);
            }
        } catch(error) {
            efectosJarvis.efectoError();
            mostrarEstado('statusImagen', '❌ Error de conexión', true);
        }
    }

    async function obtenerClima() {
        agregarLog('Consultando clima...');
        mostrarEstado('statusClima', '🌤️ Consultando clima...');
        efectosJarvis.efectoProcesando();
        
        try {
            const response = await fetch('panel_master.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: 'action=clima'
            });
            
            const data = await response.json();
            
            if(data.ok) {
                efectosJarvis.efectoRespuesta();
                const textoClima = data.clima.replace(/<[^>]*>/g, '').replace(/\n/g, ' | ');
                mostrarEstado('statusClima', `✅ ${textoClima}`);
                agregarLog(`✅ Clima: ${textoClima}`);
            } else {
                efectosJarvis.efectoError();
                mostrarEstado('statusClima', '❌ Error obteniendo clima', true);
            }
        } catch(error) {
            efectosJarvis.efectoError();
            mostrarEstado('statusClima', '❌ Error de conexión', true);
        }
    }

    // ===== INICIALIZACIÓN =====
    document.addEventListener('DOMContentLoaded', function() {
        agregarLog('🤖 JARVIS activado - Temas profesionales cargados');
        
        // CARGAR TEMA GUARDADO
        const temaGuardado = localStorage.getItem('temaJarvis');
        if (temaGuardado === 'oscuro') {
            activarModoOscuro();
        } else if (temaGuardado === 'claro') {
            activarModoClaro();
        }
        // Si es 'principal' o no hay, se queda con el tema por defecto (:root)
        
        // CARGAR CONFIGURACIONES DE VOZ
        const vozGuardada = localStorage.getItem('vozSeleccionadaJarvis');
        if (vozGuardada) {
            try {
                const vozData = JSON.parse(vozGuardada);
                cargarVoces().then(() => {
                    const vozEncontrada = vocesDisponibles.find(voz => 
                        voz.name === vozData.name && voz.lang === vozData.lang
                    );
                    if (vozEncontrada) {
                        vozSeleccionada = vozEncontrada;
                        actualizarSelectorVoces();
                    }
                });
            } catch (e) {
                console.log('Error cargando voz guardada:', e);
            }
        }
        
        const ultimaRespuestaGuardada = localStorage.getItem('ultimaRespuestaJarvis');
        if (ultimaRespuestaGuardada) {
            ultimaRespuesta = ultimaRespuestaGuardada;
        }
        
        // Cargar voces al iniciar
        cargarVoces().then(() => {
            agregarLog('🔊 Voces TTS cargadas correctamente');
        });
        
        // Micrófono
        document.getElementById('micButton').addEventListener('click', function() {
            if (this.classList.contains('listening')) {
                stopListening();
                document.getElementById('transcript').textContent = 'Micrófono desactivado';
            } else {
                startListening();
            }
        });
        
        // Enter en inputs
        document.getElementById('promptImagen').addEventListener('keypress', function(e) {
            if(e.key === 'Enter') generarImagen();
        });
        
        document.getElementById('preguntaIA').addEventListener('keypress', function(e) {
            if(e.key === 'Enter') chatIA();
        });
        
        // Cargar voces TTS cuando estén disponibles
        speechSynthesis.onvoiceschanged = function() {
            cargarVoces();
        };
    });
</script>
</body>
</html>